<?php
/**
 * API for Telegram Mini App - Standalone Version
 */

// Disable ALL PHP errors completely
@ini_set('display_errors', '0');
@ini_set('display_startup_errors', '0');
@ini_set('log_errors', '0');
@error_reporting(0);

// Start output buffering to catch any accidental output
ob_start();

// Set headers
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, GET, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

// Handle preflight
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    ob_end_clean();
    exit(0);
}

// Response helper
function sendResponse($success, $data = [], $message = '') {
    // Clear any previous output
    while (ob_get_level()) {
        ob_end_clean();
    }
    
    // Start fresh output
    ob_start();
    echo json_encode([
        'success' => $success,
        'message' => $message,
        'data' => $data
    ], JSON_UNESCAPED_UNICODE);
    ob_end_flush();
    exit;
}

// Helper function to get file type from MIME type
function getFileTypeFromMime($mimeType) {
    if (strpos($mimeType, 'image/') === 0) return 'image';
    if (strpos($mimeType, 'video/') === 0) return 'video';
    if (strpos($mimeType, 'audio/') === 0) return 'audio';
    if (strpos($mimeType, 'application/pdf') === 0) return 'pdf';
    if (strpos($mimeType, 'application/zip') === 0 || strpos($mimeType, 'application/x-rar') === 0) return 'archive';
    return 'document';
}

// Get request data
$input = @file_get_contents('php://input');
$request = @json_decode($input, true);

if (!$request || !isset($request['action'])) {
    sendResponse(false, [], 'Invalid request');
}

$action = $request['action'];

// Handle actions
if ($action === 'getUserData') {
    if (!isset($request['user_id'])) {
        sendResponse(false, [], 'User ID required');
    }
    
    $userId = intval($request['user_id']);
    
    // Get real user data from users.json
    $usersFile = dirname(__DIR__) . '/data/users.json';
    $user = null;
    
    if (file_exists($usersFile)) {
        $usersContent = file_get_contents($usersFile);
        $users = json_decode($usersContent, true) ?: [];
        
        // Find user
        foreach ($users as $u) {
            if ($u['user_id'] == $userId) {
                $user = $u;
                break;
            }
        }
    }
    
    // If user not found, create sample data
    if (!$user) {
        $user = [
            'user_id' => $userId,
            'first_name' => 'کاربر',
            'last_name' => 'نمونه',
            'username' => 'sample_user',
            'join_date' => date('Y-m-d H:i:s'),
            'download_count' => 0,
            'is_blocked' => false,
            'warnings' => [],
            'warning_count' => 0
        ];
    }
    
    // Ensure warnings array exists
    if (!isset($user['warnings'])) {
        $user['warnings'] = [];
    }
    
    // Ensure warning count
    $user['warning_count'] = count($user['warnings']);
    
    // Get user's appeal if exists
    $appealsFile = dirname(__DIR__) . '/data/appeals.json';
    $userAppeal = null;
    
    if (file_exists($appealsFile)) {
        $appealsContent = file_get_contents($appealsFile);
        $appeals = json_decode($appealsContent, true) ?: [];
        
        // Find the latest appeal for this user
        foreach (array_reverse($appeals) as $appeal) {
            if ($appeal['user_id'] == $userId) {
                $userAppeal = $appeal;
                break;
            }
        }
    }
    
    $user['appeal'] = $userAppeal;
    
    // Get downloaded files for regular users (not admin)
    $filesFile = dirname(__DIR__) . '/data/files.json';
    $userFiles = [];
    
    if (file_exists($filesFile)) {
        $filesContent = file_get_contents($filesFile);
        $allFiles = json_decode($filesContent, true) ?: [];
        
        // Get user's downloaded files list
        $downloadedFileIds = $user['downloaded_files'] ?? [];
        
        // Get files that this user downloaded
        foreach ($allFiles as $file) {
            if (in_array($file['file_id'] ?? '', $downloadedFileIds)) {
                $userFiles[] = [
                    'id' => $file['file_id'] ?? '',
                    'file_name' => $file['filename'] ?? '',
                    'title' => $file['title'] ?? '',
                    'description' => '',
                    'file_type' => getFileTypeFromMime($file['mime_type'] ?? ''),
                    'file_size' => $file['size'] ?? 0,
                    'upload_date' => $file['upload_date'] ?? '',
                    'average_rating' => 0,
                    'total_ratings' => 0,
                    'total_downloads' => $file['download_count'] ?? 0
                ];
            }
        }
    }
    
    sendResponse(true, [
        'user' => $user,
        'files' => $userFiles
    ]);
} elseif ($action === 'getBroadcastMessages') {
    handleGetBroadcastMessages($request);
} elseif ($action === 'submitAppeal') {
    handleSubmitAppeal($request);
} else {
    sendResponse(false, [], 'Unknown action');
}

function handleSubmitAppeal($request) {
    if (!isset($request['user_id']) || !isset($request['appeal_text'])) {
        sendResponse(false, [], 'Missing required fields');
    }
    
    $userId = intval($request['user_id']);
    $appealText = trim($request['appeal_text']);
    $userInfo = $request['user_info'] ?? [];
    
    if (empty($appealText)) {
        sendResponse(false, [], 'Appeal text is required');
    }
    
    // Save appeal to file
    $appealsFile = dirname(__DIR__) . '/data/appeals.json';
    $appeals = [];
    
    if (file_exists($appealsFile)) {
        $content = file_get_contents($appealsFile);
        $appeals = json_decode($content, true) ?: [];
    }
    
    // Check if user already has an appeal
    $existingAppealIndex = null;
    foreach ($appeals as $index => $existingAppeal) {
        if ($existingAppeal['user_id'] == $userId) {
            $existingAppealIndex = $index;
            break;
        }
    }
    
    // Create appeal record
    $appeal = [
        'id' => $existingAppealIndex !== null ? $appeals[$existingAppealIndex]['id'] : uniqid('appeal_'),
        'user_id' => $userId,
        'first_name' => $userInfo['first_name'] ?? 'کاربر',
        'last_name' => $userInfo['last_name'] ?? '',
        'username' => $userInfo['username'] ?? '',
        'appeal_text' => $appealText,
        'date' => date('Y-m-d H:i:s'),
        'timestamp' => time(),
        'status' => 'pending' // pending, reviewed, accepted, rejected
    ];
    
    // Update or add appeal
    if ($existingAppealIndex !== null) {
        $appeals[$existingAppealIndex] = $appeal;
    } else {
        $appeals[] = $appeal;
    }
    
    // Save to file
    $dataDir = dirname(__DIR__) . '/data';
    if (!is_dir($dataDir)) {
        @mkdir($dataDir, 0755, true);
    }
    
    file_put_contents($appealsFile, json_encode($appeals, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
    
    // Try to send to admin via Telegram (if bot config exists)
    try {
        $configPath = dirname(__DIR__) . '/config.php';
        if (file_exists($configPath)) {
            require_once $configPath;
            
            if (defined('BOT_TOKEN') && defined('ADMIN_ID')) {
                $userName = trim($userInfo['first_name'] . ' ' . ($userInfo['last_name'] ?? ''));
                $username = $userInfo['username'] ? '@' . $userInfo['username'] : 'بدون نام کاربری';
                
                $message = "🔔 اعتراض جدید از کاربر بلاک شده\n\n";
                $message .= "👤 نام: " . $userName . "\n";
                $message .= "🆔 Username: " . $username . "\n";
                $message .= "🆔 User ID: " . $userId . "\n\n";
                $message .= "📝 متن اعتراض:\n" . $appealText . "\n\n";
                $message .= "📅 تاریخ: " . date('Y/m/d H:i');
                
                $url = "https://api.telegram.org/bot" . BOT_TOKEN . "/sendMessage";
                $data = [
                    'chat_id' => ADMIN_ID,
                    'text' => $message,
                    'parse_mode' => 'HTML'
                ];
                
                $ch = curl_init($url);
                curl_setopt($ch, CURLOPT_POST, true);
                curl_setopt($ch, CURLOPT_POSTFIELDS, $data);
                curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
                curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
                curl_exec($ch);
                curl_close($ch);
            }
        }
    } catch (Exception $e) {
        // Ignore telegram send errors
    }
    
    sendResponse(true, ['appeal_id' => $appeal['id']], 'Appeal submitted successfully');
}

/**
 * Get broadcast messages for user
 */
function handleGetBroadcastMessages($request) {
    if (!isset($request['user_id'])) {
        sendResponse(false, [], 'User ID required');
    }
    
    $userId = intval($request['user_id']);
    
    // Get activity log to find broadcast messages
    $activityFile = dirname(__DIR__) . '/data/activity.json';
    $broadcasts = [];
    
    if (file_exists($activityFile)) {
        $activityContent = file_get_contents($activityFile);
        $activities = json_decode($activityContent, true) ?: [];
        
        // Filter broadcast activities
        foreach (array_reverse($activities) as $activity) {
            if (isset($activity['type']) && $activity['type'] === 'broadcast_sent') {
                $broadcasts[] = [
                    'id' => $activity['id'] ?? uniqid(),
                    'message' => $activity['details']['message'] ?? '',
                    'sent_at' => $activity['timestamp'] ?? '',
                    'formatted_date' => isset($activity['timestamp']) ? date('Y/m/d H:i', strtotime($activity['timestamp'])) : '',
                    'admin_id' => $activity['user_id'] ?? 0
                ];
            }
        }
    }
    
    sendResponse(true, ['broadcasts' => $broadcasts]);
}
